/*!
 * Copyright 2017 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
'use strict';
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const bun_1 = __importDefault(require("bun"));
const deep_equal_1 = __importDefault(require("deep-equal"));
const extend_1 = __importDefault(require("extend"));
const is_1 = __importDefault(require("is"));
const through2_1 = __importDefault(require("through2"));
const order_1 = require("./order");
const logger_1 = require("./logger");
const document_1 = require("./document");
const document_change_1 = require("./document-change");
const watch_1 = require("./watch");
const write_batch_1 = require("./write-batch");
const timestamp_1 = require("./timestamp");
const path_1 = require("./path");
const util_1 = require("./util");
const validate_1 = require("./validate");
/*!
 * The direction of a `Query.orderBy()` clause is specified as 'desc' or 'asc'
 * (descending or ascending).
 *
 * @private
 */
const directionOperators = {
    asc: 'ASCENDING',
    ASC: 'ASCENDING',
    desc: 'DESCENDING',
    DESC: 'DESCENDING',
};
/*!
 * Filter conditions in a `Query.where()` clause are specified using the
 * strings '<', '<=', '==', '>=', and '>'.
 *
 * @private
 */
const comparisonOperators = {
    '<': 'LESS_THAN',
    '<=': 'LESS_THAN_OR_EQUAL',
    '=': 'EQUAL',
    '==': 'EQUAL',
    '>': 'GREATER_THAN',
    '>=': 'GREATER_THAN_OR_EQUAL',
    'array-contains': 'ARRAY_CONTAINS'
};
/**
 * onSnapshot() callback that receives a QuerySnapshot.
 *
 * @callback querySnapshotCallback
 * @param {QuerySnapshot} snapshot - A query snapshot.
 */
/**
 * onSnapshot() callback that receives a DocumentSnapshot.
 *
 * @callback documentSnapshotCallback
 * @param {DocumentSnapshot} snapshot - A document snapshot.
 */
/**
 * onSnapshot() callback that receives an error.
 *
 * @callback errorCallback
 * @param {Error} err - An error from a listen.
 */
/**
 * A DocumentReference refers to a document location in a Firestore database
 * and can be used to write, read, or listen to the location. The document at
 * the referenced location may or may not exist. A DocumentReference can
 * also be used to create a
 * [CollectionReference]{@link CollectionReference} to a
 * subcollection.
 *
 * @class
 */
class DocumentReference {
    /**
     * @private
     * @hideconstructor
     *
     * @param {Firestore} firestore - The Firestore Database client.
     * @param {ResourcePath} path - The Path of this reference.
     */
    constructor(firestore, path) {
        this._firestore = firestore;
        this._validator = firestore._validator;
        this._referencePath = path;
    }
    /**
     * The string representation of the DocumentReference's location.
     * @private
     * @type {string}
     * @name DocumentReference#formattedName
     */
    get formattedName() {
        return this._referencePath.formattedName;
    }
    /**
     * The [Firestore]{@link Firestore} instance for the Firestore
     * database (useful for performing transactions, etc.).
     *
     * @type {Firestore}
     * @name DocumentReference#firestore
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col');
     *
     * collectionRef.add({foo: 'bar'}).then(documentReference => {
     *   let firestore = documentReference.firestore;
     *   console.log(`Root location for document is ${firestore.formattedName}`);
     * });
     */
    get firestore() {
        return this._firestore;
    }
    /**
     * A string representing the path of the referenced document (relative
     * to the root of the database).
     *
     * @type {string}
     * @name DocumentReference#path
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col');
     *
     * collectionRef.add({foo: 'bar'}).then(documentReference => {
     *   console.log(`Added document at '${documentReference.path}'`);
     * });
     */
    get path() {
        return this._referencePath.relativeName;
    }
    /**
     * The last path element of the referenced document.
     *
     * @type {string}
     * @name DocumentReference#id
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col');
     *
     * collectionRef.add({foo: 'bar'}).then(documentReference => {
     *   console.log(`Added document with name '${documentReference.id}'`);
     * });
     */
    get id() {
        return this._referencePath.id;
    }
    /**
     * A reference to the collection to which this DocumentReference belongs.
     *
     * @name DocumentReference#parent
     * @type {CollectionReference}
     * @readonly
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     * let collectionRef = documentRef.parent;
     *
     * collectionRef.where('foo', '==', 'bar').get().then(results => {
     *   console.log(`Found ${results.size} matches in parent collection`);
     * }):
     */
    get parent() {
        return createCollectionReference(this._firestore, this._referencePath.parent());
    }
    /**
     * Returns the [ResourcePath]{@link ResourcePath} for this
     * DocumentReference.
     *
     * @private
     * @type {ResourcePath}
     * @readonly
     */
    get ref() {
        return this._referencePath;
    }
    /**
     * Retrieve a document from the database. Fails the Promise if the document is
     * not found.
     *
     * @returns {Promise.<DocumentSnapshot>} A Promise resolved with a
     * DocumentSnapshot for the retrieved document on success. For missing
     * documents, DocumentSnapshot.exists will be false. If the get() fails for
     * other reasons, the Promise will be rejected.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * documentRef.get().then(documentSnapshot => {
     *   if (documentSnapshot.exists) {
     *     console.log('Document retrieved successfully.');
     *   }
     * });
     */
    get() {
        return this._firestore.getAll([this]).then(result => {
            return result[0];
        });
    }
    /**
     * Gets a [CollectionReference]{@link CollectionReference} instance
     * that refers to the collection at the specified path.
     *
     * @param {string} collectionPath - A slash-separated path to a collection.
     * @returns {CollectionReference} A reference to the new
     * subcollection.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     * let subcollection = documentRef.collection('subcollection');
     * console.log(`Path to subcollection: ${subcollection.path}`);
     */
    collection(collectionPath) {
        this._validator.isResourcePath('collectionPath', collectionPath);
        let path = this._referencePath.append(collectionPath);
        if (!path.isCollection) {
            throw new Error(`Argument "collectionPath" must point to a collection, but was "${collectionPath}". Your path does not contain an odd number of components.`);
        }
        return createCollectionReference(this._firestore, path);
    }
    /**
     * Fetches the subcollections that are direct children of this document.
     *
     * @returns {Promise.<Array.<CollectionReference>>} A Promise that resolves
     * with an array of CollectionReferences.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * documentRef.getCollections().then(collections => {
     *   for (let collection of collections) {
     *     console.log(`Found subcollection with id: ${collection.id}`);
     *   }
     * });
     */
    getCollections() {
        const request = {
            parent: this._referencePath.formattedName,
        };
        return this._firestore.request('listCollectionIds', request, util_1.requestTag())
            .then(collectionIds => {
            let collections = [];
            // We can just sort this list using the default comparator since it
            // will only contain collection ids.
            collectionIds.sort();
            for (let collectionId of collectionIds) {
                collections.push(this.collection(collectionId));
            }
            return collections;
        });
    }
    /**
     * Create a document with the provided object values. This will fail the write
     * if a document exists at its location.
     *
     * @param {DocumentData} data - An object that contains the fields and data to
     * serialize as the document.
     * @returns {Promise.<WriteResult>} A Promise that resolves with the
     * write time of this create.
     *
     * @example
     * let documentRef = firestore.collection('col').doc();
     *
     * documentRef.create({foo: 'bar'}).then((res) => {
     *   console.log(`Document created at ${res.updateTime}`);
     * }).catch((err) => {
     *   console.log(`Failed to create document: ${err}`);
     * });
     */
    create(data) {
        let writeBatch = new write_batch_1.WriteBatch(this._firestore);
        return writeBatch.create(this, data).commit().then(writeResults => {
            return Promise.resolve(writeResults[0]);
        });
    }
    /**
     * Deletes the document referred to by this `DocumentReference`.
     *
     * A delete for a non-existing document is treated as a success (unless
     * lastUptimeTime is provided).
     *
     * @param {Precondition=} precondition - A precondition to enforce for this
     * delete.
     * @param {Timestamp=} precondition.lastUpdateTime If set, enforces that the
     * document was last updated at lastUpdateTime. Fails the delete if the
     * document was last updated at a different time.
     * @returns {Promise.<WriteResult>} A Promise that resolves with the
     * delete time.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * documentRef.delete().then(() => {
     *   console.log('Document successfully deleted.');
     * });
     */
    delete(precondition) {
        let writeBatch = new write_batch_1.WriteBatch(this._firestore);
        return writeBatch.delete(this, precondition).commit().then(writeResults => {
            return Promise.resolve(writeResults[0]);
        });
    }
    /**
     * Writes to the document referred to by this DocumentReference. If the
     * document does not yet exist, it will be created. If you pass
     * [SetOptions]{@link SetOptions}, the provided data can be merged into an
     * existing document.
     *
     * @param {DocumentData} data - A map of the fields and values for the
     * document.
     * @param {SetOptions=} options - An object to configure the set behavior.
     * @param {boolean=} options.merge - If true, set() merges the values
     * specified in its data argument. Fields omitted from this set() call
     * remain untouched.
     * @param {Array.<string|FieldPath>=} options.mergeFields - If provided,
     * set() only replaces the specified field paths. Any field path that is not
     * specified is ignored and remains untouched.
     * @returns {Promise.<WriteResult>} A Promise that resolves with the
     * write time of this set.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * documentRef.set({foo: 'bar'}).then(res => {
     *   console.log(`Document written at ${res.updateTime}`);
     * });
     */
    set(data, options) {
        let writeBatch = new write_batch_1.WriteBatch(this._firestore);
        return writeBatch.set(this, data, options).commit().then(writeResults => {
            return Promise.resolve(writeResults[0]);
        });
    }
    /**
     * Updates fields in the document referred to by this DocumentReference.
     * If the document doesn't yet exist, the update fails and the returned
     * Promise will be rejected.
     *
     * The update() method accepts either an object with field paths encoded as
     * keys and field values encoded as values, or a variable number of arguments
     * that alternate between field paths and field values.
     *
     * A Precondition restricting this update can be specified as the last
     * argument.
     *
     * @param {UpdateData|string|FieldPath} dataOrField - An object
     * containing the fields and values with which to update the document
     * or the path of the first field to update.
     * @param {
     * ...(*|string|FieldPath|Precondition)} preconditionOrValues -
     * An alternating list of field paths and values to update or a Precondition
     * to restrict this update.
     * @returns Promise.<WriteResult> A Promise that resolves once the
     * data has been successfully written to the backend.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * documentRef.update({foo: 'bar'}).then(res => {
     *   console.log(`Document updated at ${res.updateTime}`);
     * });
     */
    update(dataOrField, preconditionOrValues) {
        this._validator.minNumberOfArguments('update', arguments, 1);
        let writeBatch = new write_batch_1.WriteBatch(this._firestore);
        preconditionOrValues = Array.prototype.slice.call(arguments, 1);
        return writeBatch.update
            .apply(writeBatch, [this, dataOrField].concat(preconditionOrValues))
            .commit()
            .then(writeResults => {
            return Promise.resolve(writeResults[0]);
        });
    }
    /**
     * Attaches a listener for DocumentSnapshot events.
     *
     * @param {documentSnapshotCallback} onNext - A callback to be called every
     * time a new `DocumentSnapshot` is available.
     * @param {errorCallback=} onError - A callback to be called if the listen
     * fails or is cancelled. No further callbacks will occur. If unset, errors
     * will be logged to the console.
     *
     * @returns {function()} An unsubscribe function that can be called to cancel
     * the snapshot listener.
     *
     * @example
     * let documentRef = firestore.doc('col/doc');
     *
     * let unsubscribe = documentRef.onSnapshot(documentSnapshot => {
     *   if (documentSnapshot.exists) {
     *     console.log(documentSnapshot.data());
     *   }
     * }, err => {
     *   console.log(`Encountered error: ${err}`);
     * });
     *
     * // Remove this listener.
     * unsubscribe();
     */
    onSnapshot(onNext, onError) {
        this._validator.isFunction('onNext', onNext);
        this._validator.isOptionalFunction('onError', onError);
        if (!is_1.default.defined(onError)) {
            onError = console.error;
        }
        let watch = watch_1.Watch.forDocument(this);
        return watch.onSnapshot((readTime, size, docs) => {
            for (let document of docs()) {
                if (document.ref.path === this.path) {
                    onNext(document);
                    return;
                }
            }
            // The document is missing.
            let document = new document_1.DocumentSnapshot.Builder();
            document.ref =
                new DocumentReference(this._firestore, this._referencePath);
            document.readTime = readTime;
            onNext(document.build());
        }, onError);
    }
    /**
     * Returns true if this `DocumentReference` is equal to the provided value.
     *
     * @param {*} other The value to compare against.
     * @return {boolean} true if this `DocumentReference` is equal to the provided
     * value.
     */
    isEqual(other) {
        return (this === other ||
            (is_1.default.instanceof(other, DocumentReference) &&
                this._firestore === other._firestore &&
                this._referencePath.isEqual(other._referencePath)));
    }
    /**
     * Converts this DocumentReference to the Firestore Proto representation.
     *
     * @private
     */
    toProto() {
        return {
            referenceValue: this.formattedName
        };
    }
}
exports.DocumentReference = DocumentReference;
/**
 * A Query order-by field.
 *
 * @private
 * @class
 */
class FieldOrder {
    /**
     * @private
     * @hideconstructor
     *
     * @param {FieldPath} field - The name of a document field (member)
     * on which to order query results.
     * @param {string=} direction One of 'ASCENDING' (default) or 'DESCENDING' to
     * set the ordering direction to ascending or descending, respectively.
     */
    constructor(field, direction) {
        this._field = field;
        this._direction = direction || directionOperators.ASC;
    }
    /**
     * The path of the field on which to order query results.
     *
     * @private
     * @type {FieldPath}
     */
    get field() {
        return this._field;
    }
    /**
     * One of 'ASCENDING' (default) or 'DESCENDING'.
     *
     * @private
     * @type {string}
     */
    get direction() {
        return this._direction;
    }
    /**
     * Generates the proto representation for this field order.
     *
     * @private
     * @returns {Object}
     */
    toProto() {
        return {
            field: {
                fieldPath: this._field.formattedName,
            },
            direction: this._direction,
        };
    }
}
/*!
 * A field constraint for a Query where clause.
 *
 * @private
 * @class
 */
class FieldFilter {
    /**
     * @private
     * @hideconstructor
     *
     * @param {FieldPath} field - The path of the property value to
     * compare.
     * @param {string} opString - A comparison operation.
     * @param {*} value The value to which to compare the
     * field for inclusion in a query.
     */
    constructor(serializer, field, opString, value) {
        this._serializer = serializer;
        this._field = field;
        this._opString = opString;
        this._value = value;
    }
    /**
     * Returns the field path of this filter.
     *
     * @private
     * @return {FieldPath}
     */
    get field() {
        return this._field;
    }
    /**
     * Returns whether this FieldFilter uses an equals comparison.
     *
     * @private
     * @return {boolean}
     */
    isInequalityFilter() {
        return this._opString === 'GREATER_THAN' ||
            this._opString === 'GREATER_THAN_OR_EQUAL' ||
            this._opString === 'LESS_THAN' ||
            this._opString === 'LESS_THAN_OR_EQUAL';
    }
    /**
     * Generates the proto representation for this field filter.
     *
     * @private
     * @returns {Object}
     */
    toProto() {
        if (typeof this._value === 'number' && isNaN(this._value)) {
            return {
                unaryFilter: {
                    field: {
                        fieldPath: this._field.formattedName,
                    },
                    op: 'IS_NAN',
                },
            };
        }
        if (this._value === null) {
            return {
                unaryFilter: {
                    field: {
                        fieldPath: this._field.formattedName,
                    },
                    op: 'IS_NULL',
                },
            };
        }
        return {
            fieldFilter: {
                field: {
                    fieldPath: this._field.formattedName,
                },
                op: this._opString,
                value: this._serializer.encodeValue(this._value),
            },
        };
    }
}
/**
 * A QuerySnapshot contains zero or more
 * [QueryDocumentSnapshot]{@link QueryDocumentSnapshot} objects
 * representing the results of a query. The documents can be accessed as an
 * array via the [documents]{@link QuerySnapshot#documents} property
 * or enumerated using the [forEach]{@link QuerySnapshot#forEach}
 * method. The number of documents can be determined via the
 * [empty]{@link QuerySnapshot#empty} and
 * [size]{@link QuerySnapshot#size} properties.
 *
 * @class QuerySnapshot
 */
class QuerySnapshot {
    /**
     * @private
     * @hideconstructor
     *
     * @param {Query} query - The originating query.
     * @param {Timestamp} readTime - The time when this query snapshot was
     * obtained.
     * @param {number} size - The number of documents in the result set.
     * @param {function} docs - A callback returning a sorted array of documents
     * matching this query
     * @param {function} changes - A callback returning a sorted array of
     * document change events for this snapshot.
     */
    constructor(query, readTime, size, docs, changes) {
        this._query = query;
        this._validator = query.firestore._validator;
        this._readTime = readTime;
        this._size = size;
        this._docs = docs;
        this._materializedDocs = null;
        this._changes = changes;
        this._materializedChanges = null;
    }
    /**
     * The query on which you called get() or onSnapshot() in order to get this
     * QuerySnapshot.
     *
     * @type {Query}
     * @name QuerySnapshot#query
     * @readonly
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.limit(10).get().then(querySnapshot => {
     *   console.log(`Returned first batch of results`);
     *   let query = querySnapshot.query;
     *   return query.offset(10).get();
     * }).then(() => {
     *   console.log(`Returned second batch of results`);
     * });
     */
    get query() {
        return this._query;
    }
    /**
     * An array of all the documents in this QuerySnapshot.
     *
     * @type {Array.<QueryDocumentSnapshot>}
     * @name QuerySnapshot#docs
     * @readonly
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then(querySnapshot => {
     *   let docs = querySnapshot.docs;
     *   for (let doc of docs) {
     *     console.log(`Document found at path: ${doc.ref.path}`);
     *   }
     * });
     */
    get docs() {
        if (this._materializedDocs) {
            return this._materializedDocs;
        }
        this._materializedDocs = this._docs();
        this._docs = null;
        return this._materializedDocs;
    }
    /**
     * An array of all changes in this QuerySnapshot.
     *
     * @type {Array.<DocumentChange>}
     * @name QuerySnapshot#docChanges
     * @readonly
     */
    get docChanges() {
        if (this._materializedChanges) {
            return this._materializedChanges;
        }
        this._materializedChanges = this._changes();
        this._changes = null;
        return this._materializedChanges;
    }
    /**
     * True if there are no documents in the QuerySnapshot.
     *
     * @type {boolean}
     * @name QuerySnapshot#empty
     * @readonly
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then(querySnapshot => {
     *   if (querySnapshot.empty) {
     *     console.log('No documents found.');
     *   }
     * });
     */
    get empty() {
        return this._size === 0;
    }
    /**
     * The number of documents in the QuerySnapshot.
     *
     * @type {number}
     * @name QuerySnapshot#size
     * @readonly
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then(querySnapshot => {
     *   console.log(`Found ${querySnapshot.size} documents.`);
     * });
     */
    get size() {
        return this._size;
    }
    /**
     * The time this query snapshot was obtained.
     *
     * @type {Timestamp}
     * @name QuerySnapshot#readTime
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then((querySnapshot) => {
     *   let readTime = querySnapshot.readTime;
     *   console.log(`Query results returned at '${readTime.toDate()}'`);
     * });
     */
    get readTime() {
        return this._readTime;
    }
    /**
     * Enumerates all of the documents in the QuerySnapshot.
     *
     * @param {function} callback - A callback to be called with a
     * [QueryDocumentSnapshot]{@link QueryDocumentSnapshot} for each document in
     * the snapshot.
     * @param {*=} thisArg The `this` binding for the callback..
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Document found at path: ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    forEach(callback, thisArg) {
        this._validator.isFunction('callback', callback);
        for (let doc of this.docs) {
            callback.call(thisArg, doc);
        }
    }
    /**
     * Returns true if the document data in this `QuerySnapshot` is equal to the
     * provided value.
     *
     * @param {*} other The value to compare against.
     * @return {boolean} true if this `QuerySnapshot` is equal to the provided
     * value.
     */
    isEqual(other) {
        // Since the read time is different on every query read, we explicitly
        // ignore all metadata in this comparison.
        if (this === other) {
            return true;
        }
        if (!is_1.default.instanceof(other, QuerySnapshot)) {
            return false;
        }
        if (this._size !== other._size) {
            return false;
        }
        if (!this._query.isEqual(other._query)) {
            return false;
        }
        if (this._materializedDocs && !this._materializedChanges) {
            // If we have only materialized the documents, we compare them first.
            return (isArrayEqual(this.docs, other.docs) &&
                isArrayEqual(this.docChanges, other.docChanges));
        }
        // Otherwise, we compare the changes first as we expect there to be fewer.
        return (isArrayEqual(this.docChanges, other.docChanges) &&
            isArrayEqual(this.docs, other.docs));
    }
}
exports.QuerySnapshot = QuerySnapshot;
/**
 * A Query refers to a query which you can read or stream from. You can also
 * construct refined Query objects by adding filters and ordering.
 *
 * @class Query
 */
class Query {
    /**
     * @private
     * @hideconstructor
     *
     * @param {Firestore} firestore - The Firestore Database client.
     * @param {ResourcePath} path Path of the collection to be queried.
     * @param {Array.<FieldOrder>=} fieldOrders - Sequence of fields to
     * control the order of results.
     * @param {Array.<FieldFilter>=} fieldFilters - Sequence of fields
     * constraining the results of the query.
     * @param {object=} queryOptions Additional query options.
     */
    constructor(firestore, path, fieldFilters, fieldOrders, queryOptions) {
        this._firestore = firestore;
        this._serializer = firestore._serializer;
        this._validator = firestore._validator;
        this._referencePath = path;
        this._fieldFilters = fieldFilters || [];
        this._fieldOrders = fieldOrders || [];
        this._queryOptions = queryOptions || {};
    }
    /**
     * Detects the argument type for Firestore cursors.
     *
     * @private
     * @param {Array.<DocumentSnapshot|*>} fieldValuesOrDocumentSnapshot - A
     * snapshot of the document or a set of field values.
     * @returns {boolean} 'true' if the input is a single DocumentSnapshot..
     */
    static _isDocumentSnapshot(fieldValuesOrDocumentSnapshot) {
        return (fieldValuesOrDocumentSnapshot.length === 1 &&
            is_1.default.instance(fieldValuesOrDocumentSnapshot[0], document_1.DocumentSnapshot));
    }
    /**
     * Extracts field values from the DocumentSnapshot based on the provided
     * field order.
     *
     * @private
     * @param {DocumentSnapshot} documentSnapshot - The document to extract the
     * fields from.
     * @param {Array.<FieldOrder>} fieldOrders - The field order that defines what
     * fields we should extract.
     * @return {Array.<*>} The field values to use.
     * @private
     */
    static _extractFieldValues(documentSnapshot, fieldOrders) {
        let fieldValues = [];
        for (let fieldOrder of fieldOrders) {
            if (path_1.FieldPath._DOCUMENT_ID.isEqual(fieldOrder.field)) {
                fieldValues.push(documentSnapshot.ref);
            }
            else {
                let fieldValue = documentSnapshot.get(fieldOrder.field);
                if (is_1.default.undefined(fieldValue)) {
                    throw new Error(`Field '${fieldOrder
                        .field}' is missing in the provided DocumentSnapshot. Please provide a ` +
                        'document that contains values for all specified orderBy() and ' +
                        'where() constraints.');
                }
                else {
                    fieldValues.push(fieldValue);
                }
            }
        }
        return fieldValues;
    }
    /**
     * The string representation of the Query's location.
     * @private
     * @type {string}
     * @name Query#formattedName
     */
    get formattedName() {
        return this._referencePath.formattedName;
    }
    /**
     * The [Firestore]{@link Firestore} instance for the Firestore
     * database (useful for performing transactions, etc.).
     *
     * @type {Firestore}
     * @name Query#firestore
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col');
     *
     * collectionRef.add({foo: 'bar'}).then(documentReference => {
     *   let firestore = documentReference.firestore;
     *   console.log(`Root location for document is ${firestore.formattedName}`);
     * });
     */
    get firestore() {
        return this._firestore;
    }
    /**
     * Creates and returns a new [Query]{@link Query} with the additional filter
     * that documents must contain the specified field and that its value should
     * satisfy the relation constraint provided.
     *
     * Returns a new Query that constrains the value of a Document property.
     *
     * This function returns a new (immutable) instance of the Query (rather than
     * modify the existing instance) to impose the filter.
     *
     * @param {string|FieldPath} fieldPath - The name of a property
     * value to compare.
     * @param {string} opStr - A comparison operation in the form of a string
     * (e.g., "<").
     * @param {*} value - The value to which to compare the field for inclusion in
     * a query.
     * @returns {Query} The created Query.
     *
     * @example
     * let collectionRef = firestore.collection('col');
     *
     * collectionRef.where('foo', '==', 'bar').get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    where(fieldPath, opStr, value) {
        this._validator.isFieldPath('fieldPath', fieldPath);
        this._validator.isQueryComparison('opStr', opStr, value);
        this._validator.isQueryValue('value', value, {
            allowDeletes: 'none',
            allowTransforms: false,
        });
        if (this._queryOptions.startAt || this._queryOptions.endAt) {
            throw new Error('Cannot specify a where() filter after calling startAt(), ' +
                'startAfter(), endBefore() or endAt().');
        }
        fieldPath = path_1.FieldPath.fromArgument(fieldPath);
        if (path_1.FieldPath._DOCUMENT_ID.isEqual(fieldPath)) {
            value = this._convertReference(value);
        }
        let combinedFilters = this._fieldFilters.concat(new FieldFilter(this._serializer, fieldPath, comparisonOperators[opStr], value));
        return new Query(this._firestore, this._referencePath, combinedFilters, this._fieldOrders, this._queryOptions);
    }
    /**
     * Creates and returns a new [Query]{@link Query} instance that applies a
     * field mask to the result and returns only the specified subset of fields.
     * You can specify a list of field paths to return, or use an empty list to
     * only return the references of matching documents.
     *
     * This function returns a new (immutable) instance of the Query (rather than
     * modify the existing instance) to impose the field mask.
     *
     * @param {...(string|FieldPath)} fieldPaths - The field paths to
     * return.
     * @returns {Query} The created Query.
     *
     * @example
     * let collectionRef = firestore.collection('col');
     * let documentRef = collectionRef.doc('doc');
     *
     * return documentRef.set({x:10, y:5}).then(() => {
     *   return collectionRef.where('x', '>', 5).select('y').get();
     * }).then((res) => {
     *   console.log(`y is ${res.docs[0].get('y')}.`);
     * });
     */
    select(fieldPaths) {
        fieldPaths = [].slice.call(arguments);
        let result = [];
        if (fieldPaths.length === 0) {
            result.push({ fieldPath: path_1.FieldPath._DOCUMENT_ID.formattedName });
        }
        else {
            for (let i = 0; i < fieldPaths.length; ++i) {
                this._validator.isFieldPath(i, fieldPaths[i]);
                result.push({
                    fieldPath: path_1.FieldPath.fromArgument(fieldPaths[i]).formattedName,
                });
            }
        }
        let options = extend_1.default(true, {}, this._queryOptions);
        options.selectFields = { fields: result };
        return new Query(this._firestore, this._referencePath, this._fieldFilters, this._fieldOrders, options);
    }
    /**
     * Creates and returns a new [Query]{@link Query} that's additionally sorted
     * by the specified field, optionally in descending order instead of
     * ascending.
     *
     * This function returns a new (immutable) instance of the Query (rather than
     * modify the existing instance) to impose the field mask.
     *
     * @param {string|FieldPath} fieldPath - The field to sort by.
     * @param {string=} directionStr - Optional direction to sort by ('asc' or
     * 'desc'). If not specified, order will be ascending.
     * @returns {Query} The created Query.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '>', 42);
     *
     * query.orderBy('foo', 'desc').get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    orderBy(fieldPath, directionStr) {
        this._validator.isFieldPath('fieldPath', fieldPath);
        this._validator.isOptionalFieldOrder('directionStr', directionStr);
        if (this._queryOptions.startAt || this._queryOptions.endAt) {
            throw new Error('Cannot specify an orderBy() constraint after calling ' +
                'startAt(), startAfter(), endBefore() or endAt().');
        }
        let newOrder = new FieldOrder(path_1.FieldPath.fromArgument(fieldPath), directionOperators[directionStr]);
        let combinedOrders = this._fieldOrders.concat(newOrder);
        return new Query(this._firestore, this._referencePath, this._fieldFilters, combinedOrders, this._queryOptions);
    }
    /**
     * Creates and returns a new [Query]{@link Query} that's additionally limited
     * to only return up to the specified number of documents.
     *
     * This function returns a new (immutable) instance of the Query (rather than
     * modify the existing instance) to impose the limit.
     *
     * @param {number} limit - The maximum number of items to return.
     * @returns {Query} The created Query.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '>', 42);
     *
     * query.limit(1).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    limit(limit) {
        this._validator.isInteger('limit', limit);
        let options = extend_1.default(true, {}, this._queryOptions);
        options.limit = limit;
        return new Query(this._firestore, this._referencePath, this._fieldFilters, this._fieldOrders, options);
    }
    /**
     * Specifies the offset of the returned results.
     *
     * This function returns a new (immutable) instance of the
     * [Query]{@link Query} (rather than modify the existing instance)
     * to impose the offset.
     *
     * @param {number} offset - The offset to apply to the Query results
     * @returns {Query} The created Query.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '>', 42);
     *
     * query.limit(10).offset(20).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    offset(offset) {
        this._validator.isInteger('offset', offset);
        let options = extend_1.default(true, {}, this._queryOptions);
        options.offset = offset;
        return new Query(this._firestore, this._referencePath, this._fieldFilters, this._fieldOrders, options);
    }
    /**
     * Returns true if this `Query` is equal to the provided value.
     *
     * @param {*} other The value to compare against.
     * @return {boolean} true if this `Query` is equal to the provided value.
     */
    isEqual(other) {
        if (this === other) {
            return true;
        }
        return (is_1.default.instanceof(other, Query) &&
            this._referencePath.isEqual(other._referencePath) &&
            deep_equal_1.default(this._fieldFilters, other._fieldFilters, { strict: true }) &&
            deep_equal_1.default(this._fieldOrders, other._fieldOrders, { strict: true }) &&
            deep_equal_1.default(this._queryOptions, other._queryOptions, { strict: true }));
    }
    /**
     * Computes the backend ordering semantics for DocumentSnapshot cursors.
     *
     * @private
     * @param {Array.<DocumentSnapshot|*>} cursorValuesOrDocumentSnapshot - The
     * snapshot of the document or the set of field values to use as the boundary.
     * @returns {Array.<FieldOrder>} The implicit ordering semantics.
     */
    _createImplicitOrderBy(cursorValuesOrDocumentSnapshot) {
        if (!Query._isDocumentSnapshot(cursorValuesOrDocumentSnapshot)) {
            return this._fieldOrders;
        }
        let fieldOrders = this._fieldOrders.slice();
        let hasDocumentId = false;
        if (fieldOrders.length === 0) {
            // If no explicit ordering is specified, use the first inequality to
            // define an implicit order.
            for (let fieldFilter of this._fieldFilters) {
                if (fieldFilter.isInequalityFilter()) {
                    fieldOrders.push(new FieldOrder(fieldFilter.field, 'ASCENDING'));
                    break;
                }
            }
        }
        else {
            for (let fieldOrder of fieldOrders) {
                if (path_1.FieldPath._DOCUMENT_ID.isEqual(fieldOrder.field)) {
                    hasDocumentId = true;
                }
            }
        }
        if (!hasDocumentId) {
            // Add implicit sorting by name, using the last specified direction.
            let lastDirection = fieldOrders.length === 0 ?
                directionOperators.ASC :
                fieldOrders[fieldOrders.length - 1].direction;
            fieldOrders.push(new FieldOrder(path_1.FieldPath.documentId(), lastDirection));
        }
        return fieldOrders;
    }
    /**
     * Builds a Firestore 'Position' proto message.
     *
     * @private
     * @param {Array.<FieldOrder>} fieldOrders - The field orders to use for this
     * cursor.
     * @param {Array.<DocumentSnapshot|*>} cursorValuesOrDocumentSnapshot - The
     * snapshot of the document or the set of field values to use as the
     * boundary.
     * @param before - Whether the query boundary lies just before or after the
     * provided data.
     * @returns {Object} The proto message.
     */
    _createCursor(fieldOrders, cursorValuesOrDocumentSnapshot, before) {
        let fieldValues;
        if (Query._isDocumentSnapshot(cursorValuesOrDocumentSnapshot)) {
            fieldValues = Query._extractFieldValues(cursorValuesOrDocumentSnapshot[0], fieldOrders);
        }
        else {
            fieldValues = cursorValuesOrDocumentSnapshot;
        }
        if (fieldValues.length > fieldOrders.length) {
            throw new Error('Too many cursor values specified. The specified ' +
                'values must match the orderBy() constraints of the query.');
        }
        let options = {
            values: [],
        };
        if (before) {
            options.before = true;
        }
        for (let i = 0; i < fieldValues.length; ++i) {
            let fieldValue = fieldValues[i];
            if (path_1.FieldPath._DOCUMENT_ID.isEqual(fieldOrders[i].field)) {
                fieldValue = this._convertReference(fieldValue);
            }
            this._validator.isQueryValue(i, fieldValue, {
                allowDeletes: 'none',
                allowTransforms: false,
            });
            options.values.push(this._serializer.encodeValue(fieldValue));
        }
        return options;
    }
    /**
     * Validates that a value used with FieldValue.documentId() is either a
     * string or a DocumentReference that is part of the query`s result set.
     * Throws a validation error or returns a DocumentReference that can
     * directly be used in the Query.
     *
     * @param {*} reference - The value to validate.
     * @throws If the value cannot be used for this query.
     * @return {DocumentReference} If valid, returns a DocumentReference that
     * can be used with the query.
     * @private
     */
    _convertReference(reference) {
        if (is_1.default.string(reference)) {
            reference = new DocumentReference(this._firestore, this._referencePath.append(reference));
        }
        else if (is_1.default.instance(reference, DocumentReference)) {
            if (!this._referencePath.isPrefixOf(reference.ref)) {
                throw new Error(`'${reference.path}' is not part of the query result set and ` +
                    'cannot be used as a query boundary.');
            }
        }
        else {
            throw new Error('The corresponding value for FieldPath.documentId() must be a ' +
                'string or a DocumentReference.');
        }
        if (reference.ref.parent().compareTo(this._referencePath) !== 0) {
            throw new Error('Only a direct child can be used as a query boundary. ' +
                `Found: '${reference.path}'.`);
        }
        return reference;
    }
    /**
     * Creates and returns a new [Query]{@link Query} that starts at the provided
     * set of field values relative to the order of the query. The order of the
     * provided values must match the order of the order by clauses of the query.
     *
     * @param {...*|DocumentSnapshot} fieldValuesOrDocumentSnapshot - The snapshot
     * of the document the query results should start at or the field values to
     * start this query at, in order of the query's order by.
     * @returns {Query} A query with the new starting point.
     *
     * @example
     * let query = firestore.collection('col');
     *
     * query.orderBy('foo').startAt(42).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    startAt(fieldValuesOrDocumentSnapshot) {
        let options = extend_1.default(true, {}, this._queryOptions);
        fieldValuesOrDocumentSnapshot = [].slice.call(arguments);
        let fieldOrders = this._createImplicitOrderBy(fieldValuesOrDocumentSnapshot);
        options.startAt =
            this._createCursor(fieldOrders, fieldValuesOrDocumentSnapshot, true);
        return new Query(this._firestore, this._referencePath, this._fieldFilters, fieldOrders, options);
    }
    /**
     * Creates and returns a new [Query]{@link Query} that starts after the
     * provided set of field values relative to the order of the query. The order
     * of the provided values must match the order of the order by clauses of the
     * query.
     *
     * @param {...*|DocumentSnapshot} fieldValuesOrDocumentSnapshot - The snapshot
     * of the document the query results should start after or the field values to
     * start this query after, in order of the query's order by.
     * @returns {Query} A query with the new starting point.
     *
     * @example
     * let query = firestore.collection('col');
     *
     * query.orderBy('foo').startAfter(42).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    startAfter(fieldValuesOrDocumentSnapshot) {
        let options = extend_1.default(true, {}, this._queryOptions);
        fieldValuesOrDocumentSnapshot = [].slice.call(arguments);
        let fieldOrders = this._createImplicitOrderBy(fieldValuesOrDocumentSnapshot);
        options.startAt =
            this._createCursor(fieldOrders, fieldValuesOrDocumentSnapshot, false);
        return new Query(this._firestore, this._referencePath, this._fieldFilters, fieldOrders, options);
    }
    /**
     * Creates and returns a new [Query]{@link Query} that ends before the set of
     * field values relative to the order of the query. The order of the provided
     * values must match the order of the order by clauses of the query.
     *
     * @param {...*|DocumentSnapshot} fieldValuesOrDocumentSnapshot - The snapshot
     * of the document the query results should end before or the field values to
     * end this query before, in order of the query's order by.
     * @returns {Query} A query with the new ending point.
     *
     * @example
     * let query = firestore.collection('col');
     *
     * query.orderBy('foo').endBefore(42).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    endBefore(fieldValuesOrDocumentSnapshot) {
        let options = extend_1.default(true, {}, this._queryOptions);
        fieldValuesOrDocumentSnapshot = [].slice.call(arguments);
        let fieldOrders = this._createImplicitOrderBy(fieldValuesOrDocumentSnapshot);
        options.endAt =
            this._createCursor(fieldOrders, fieldValuesOrDocumentSnapshot, true);
        return new Query(this._firestore, this._referencePath, this._fieldFilters, fieldOrders, options);
    }
    /**
     * Creates and returns a new [Query]{@link Query} that ends at the provided
     * set of field values relative to the order of the query. The order of the
     * provided values must match the order of the order by clauses of the query.
     *
     * @param {...*|DocumentSnapshot} fieldValuesOrDocumentSnapshot - The snapshot
     * of the document the query results should end at or the field values to end
     * this query at, in order of the query's order by.
     * @returns {Query} A query with the new ending point.
     *
     * @example
     * let query = firestore.collection('col');
     *
     * query.orderBy('foo').endAt(42).get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    endAt(fieldValuesOrDocumentSnapshot) {
        let options = extend_1.default(true, {}, this._queryOptions);
        fieldValuesOrDocumentSnapshot = [].slice.call(arguments);
        let fieldOrders = this._createImplicitOrderBy(fieldValuesOrDocumentSnapshot);
        options.endAt =
            this._createCursor(fieldOrders, fieldValuesOrDocumentSnapshot, false);
        return new Query(this._firestore, this._referencePath, this._fieldFilters, fieldOrders, options);
    }
    /**
     * Executes the query and returns the results as a
     * [QuerySnapshot]{@link QuerySnapshot}.
     *
     * @returns {Promise.<QuerySnapshot>} A Promise that resolves with the results
     * of the Query.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * query.get().then(querySnapshot => {
     *   querySnapshot.forEach(documentSnapshot => {
     *     console.log(`Found document at ${documentSnapshot.ref.path}`);
     *   });
     * });
     */
    get() {
        return this._get();
    }
    /**
     * Internal get() method that accepts an optional transaction id.
     *
     * @private
     * @param {bytes=} queryOptions.transactionId - A transaction ID.
     */
    _get(queryOptions) {
        let self = this;
        let docs = [];
        return new Promise((resolve, reject) => {
            let readTime;
            self._stream(queryOptions)
                .on('error', err => {
                reject(err);
            })
                .on('data', result => {
                readTime = result.readTime;
                if (result.document) {
                    let document = result.document;
                    docs.push(document);
                }
            })
                .on('end', () => {
                resolve(new QuerySnapshot(this, readTime, docs.length, () => docs, () => {
                    let changes = [];
                    for (let i = 0; i < docs.length; ++i) {
                        changes.push(new document_change_1.DocumentChange(document_change_1.DocumentChange.ADDED, docs[i], -1, i));
                    }
                    return changes;
                }));
            });
        });
    }
    /**
     * Executes the query and streams the results as
     * [QueryDocumentSnapshots]{@link QueryDocumentSnapshot}.
     *
     * @returns {Stream.<QueryDocumentSnapshot>} A stream of
     * QueryDocumentSnapshots.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * let count = 0;
     *
     * query.stream().on('data', (documentSnapshot) => {
     *   console.log(`Found document with name '${documentSnapshot.id}'`);
     *   ++count;
     * }).on('end', () => {
     *   console.log(`Total count is ${count}`);
     * });
     */
    stream() {
        let responseStream = this._stream();
        let transform = through2_1.default.obj(function (chunk, encoding, callback) {
            // Only send chunks with documents.
            if (chunk.document) {
                this.push(chunk.document);
            }
            callback();
        });
        return bun_1.default([responseStream, transform]);
    }
    /**
     * Internal method for serializing a query to its RunQuery proto
     * representation with an optional transaction id.
     *
     * @param {bytes=} queryOptions.transactionId - A transaction ID.
     * @private
     * @returns Serialized JSON for the query.
     */
    toProto(queryOptions) {
        let reqOpts = {
            parent: this._referencePath.parent().formattedName,
            structuredQuery: {
                from: [
                    {
                        collectionId: this._referencePath.id,
                    },
                ],
            },
        };
        let structuredQuery = reqOpts.structuredQuery;
        if (this._fieldFilters.length === 1) {
            structuredQuery.where = this._fieldFilters[0].toProto();
        }
        else if (this._fieldFilters.length > 1) {
            let filters = [];
            for (let fieldFilter of this._fieldFilters) {
                filters.push(fieldFilter.toProto());
            }
            structuredQuery.where = {
                compositeFilter: {
                    op: 'AND',
                    filters: filters,
                },
            };
        }
        if (this._fieldOrders.length) {
            let orderBy = [];
            for (let fieldOrder of this._fieldOrders) {
                orderBy.push(fieldOrder.toProto());
            }
            structuredQuery.orderBy = orderBy;
        }
        if (this._queryOptions.limit) {
            structuredQuery.limit = { value: this._queryOptions.limit };
        }
        if (this._queryOptions.offset) {
            structuredQuery.offset = this._queryOptions.offset;
        }
        if (this._queryOptions.startAt) {
            structuredQuery.startAt = this._queryOptions.startAt;
        }
        if (this._queryOptions.endAt) {
            structuredQuery.endAt = this._queryOptions.endAt;
        }
        if (this._queryOptions.selectFields) {
            structuredQuery.select = this._queryOptions.selectFields;
        }
        if (queryOptions && queryOptions.transactionId) {
            reqOpts.transaction = queryOptions.transactionId;
        }
        return reqOpts;
    }
    /**
     * Internal streaming method that accepts an optional transaction id.
     *
     * @param {bytes=} queryOptions.transactionId - A transaction ID.
     * @private
     * @returns {stream} A stream of document results.
     */
    _stream(queryOptions) {
        const request = this.toProto(queryOptions);
        const tag = util_1.requestTag();
        const self = this;
        const stream = through2_1.default.obj(function (proto, enc, callback) {
            const readTime = timestamp_1.Timestamp.fromProto(proto.readTime);
            if (proto.document) {
                let document = self.firestore.snapshot_(proto.document, proto.readTime);
                this.push({ document, readTime });
            }
            else {
                this.push({ readTime });
            }
            callback();
        });
        this._firestore.readStream('runQuery', request, tag, true)
            .then(backendStream => {
            backendStream.on('error', err => {
                logger_1.logger('Query._stream', tag, 'Query failed with stream error:', err);
                stream.destroy(err);
            });
            backendStream.resume();
            backendStream.pipe(stream);
        })
            .catch(err => {
            stream.destroy(err);
        });
        return stream;
    }
    /**
     * Attaches a listener for QuerySnapshot events.
     *
     * @param {querySnapshotCallback} onNext - A callback to be called every time
     * a new [QuerySnapshot]{@link QuerySnapshot} is available.
     * @param {errorCallback=} onError - A callback to be called if the listen
     * fails or is cancelled. No further callbacks will occur.
     *
     * @returns {function()} An unsubscribe function that can be called to cancel
     * the snapshot listener.
     *
     * @example
     * let query = firestore.collection('col').where('foo', '==', 'bar');
     *
     * let unsubscribe = query.onSnapshot(querySnapshot => {
     *   console.log(`Received query snapshot of size ${querySnapshot.size}`);
     * }, err => {
     *   console.log(`Encountered error: ${err}`);
     * });
     *
     * // Remove this listener.
     * unsubscribe();
     */
    onSnapshot(onNext, onError) {
        this._validator.isFunction('onNext', onNext);
        this._validator.isOptionalFunction('onError', onError);
        if (!is_1.default.defined(onError)) {
            onError = console.error;
        }
        let watch = watch_1.Watch.forQuery(this);
        return watch.onSnapshot((readTime, size, docs, changes) => {
            onNext(new QuerySnapshot(this, readTime, size, docs, changes));
        }, onError);
    }
    /**
     * Returns a function that can be used to sort QueryDocumentSnapshots
     * according to the sort criteria of this query.
     *
     * @private
     */
    comparator() {
        return (doc1, doc2) => {
            // Add implicit sorting by name, using the last specified direction.
            let lastDirection = this._fieldOrders.length === 0 ?
                directionOperators.ASC :
                this._fieldOrders[this._fieldOrders.length - 1].direction;
            let orderBys = this._fieldOrders.concat(new FieldOrder(path_1.FieldPath._DOCUMENT_ID, lastDirection));
            for (let orderBy of orderBys) {
                let comp;
                if (path_1.FieldPath._DOCUMENT_ID.isEqual(orderBy.field)) {
                    comp = doc1.ref._referencePath.compareTo(doc2.ref._referencePath);
                }
                else {
                    const v1 = doc1.protoField(orderBy.field);
                    const v2 = doc2.protoField(orderBy.field);
                    if (!is_1.default.defined(v1) || !is_1.default.defined(v2)) {
                        throw new Error('Trying to compare documents on fields that ' +
                            'don\'t exist. Please include the fields you are ordering on ' +
                            'in your select() call.');
                    }
                    comp = order_1.compare(v1, v2);
                }
                if (comp !== 0) {
                    const direction = orderBy.direction === directionOperators.ASC ? 1 : -1;
                    return direction * comp;
                }
            }
            return 0;
        };
    }
}
exports.Query = Query;
/**
 * A CollectionReference object can be used for adding documents, getting
 * document references, and querying for documents (using the methods
 * inherited from [Query]{@link Query}).
 *
 * @class
 * @extends Query
 */
class CollectionReference extends Query {
    /**
     * @private
     * @hideconstructor
     *
     * @param {Firestore} firestore - The Firestore Database client.
     * @param {ResourcePath} path - The Path of this collection.
     */
    constructor(firestore, path) {
        super(firestore, path);
    }
    /**
     * The last path element of the referenced collection.
     *
     * @type {string}
     * @name CollectionReference#id
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col/doc/subcollection');
     * console.log(`ID of the subcollection: ${collectionRef.id}`);
     */
    get id() {
        return this._referencePath.id;
    }
    /**
     * A reference to the containing Document if this is a subcollection, else
     * null.
     *
     * @type {DocumentReference}
     * @name CollectionReference#parent
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col/doc/subcollection');
     * let documentRef = collectionRef.parent;
     * console.log(`Parent name: ${documentRef.path}`);
     */
    get parent() {
        return new DocumentReference(this._firestore, this._referencePath.parent());
    }
    /**
     * A string representing the path of the referenced collection (relative
     * to the root of the database).
     *
     * @type {string}
     * @name CollectionReference#path
     * @readonly
     *
     * @example
     * let collectionRef = firestore.collection('col/doc/subcollection');
     * console.log(`Path of the subcollection: ${collectionRef.path}`);
     */
    get path() {
        return this._referencePath.relativeName;
    }
    /**
     * Gets a [DocumentReference]{@link DocumentReference} instance that
     * refers to the document at the specified path. If no path is specified, an
     * automatically-generated unique ID will be used for the returned
     * DocumentReference.
     *
     * @param {string=} documentPath - A slash-separated path to a document.
     * @returns {DocumentReference} The `DocumentReference`
     * instance.
     *
     * @example
     * let collectionRef = firestore.collection('col');
     * let documentRefWithName = collectionRef.doc('doc');
     * let documentRefWithAutoId = collectionRef.doc();
     * console.log(`Reference with name: ${documentRefWithName.path}`);
     * console.log(`Reference with auto-id: ${documentRefWithAutoId.path}`);
     */
    doc(documentPath) {
        if (arguments.length === 0) {
            documentPath = util_1.autoId();
        }
        else {
            this._validator.isResourcePath('documentPath', documentPath);
        }
        let path = this._referencePath.append(documentPath);
        if (!path.isDocument) {
            throw new Error(`Argument "documentPath" must point to a document, but was "${documentPath}". Your path does not contain an even number of components.`);
        }
        return new DocumentReference(this._firestore, path);
    }
    /**
     * Add a new document to this collection with the specified data, assigning
     * it a document ID automatically.
     *
     * @param {DocumentData} data - An Object containing the data for the new
     * document.
     * @returns {Promise.<DocumentReference>} A Promise resolved with a
     * [DocumentReference]{@link DocumentReference} pointing to the
     * newly created document.
     *
     * @example
     * let collectionRef = firestore.collection('col');
     * collectionRef.add({foo: 'bar'}).then(documentReference => {
     *   console.log(`Added document with name: ${documentReference.id}`);
     * });
     */
    add(data) {
        this._validator.isDocument('data', data, {
            allowEmpty: true,
            allowDeletes: 'none',
            allowTransforms: true,
        });
        let documentRef = this.doc();
        return documentRef.create(data).then(() => {
            return Promise.resolve(documentRef);
        });
    }
    /**
     * Returns true if this `CollectionReference` is equal to the provided value.
     *
     * @param {*} other The value to compare against.
     * @return {boolean} true if this `CollectionReference` is equal to the
     * provided value.
     */
    isEqual(other) {
        return (this === other ||
            (is_1.default.instanceof(other, CollectionReference) && super.isEqual(other)));
    }
}
exports.CollectionReference = CollectionReference;
/*!
 * Creates a new CollectionReference. Invoked by DocumentReference to avoid
 * invalid declaration order.
 *
 * @param {Firestore} firestore - The Firestore Database client.
 * @param {ResourcePath} path - The path of this collection.
 * @returns {CollectionReference}
 */
function createCollectionReference(firestore, path) {
    return new CollectionReference(firestore, path);
}
/*!
 * Validates the input string as a field order direction.
 *
 * @param {string=} str Order direction to validate.
 * @throws {Error} when the direction is invalid
 */
function validateFieldOrder(str) {
    if (!is_1.default.string(str) || !is_1.default.defined(directionOperators[str])) {
        throw new Error('Order must be one of "asc" or "desc".');
    }
    return true;
}
exports.validateFieldOrder = validateFieldOrder;
/*!
 * Validates the input string as a field comparison operator.
 *
 * @param {string} str Field comparison operator to validate.
 * @param {*} val Value that is used in the filter.
 * @throws {Error} when the comparison operation is invalid
 */
function validateComparisonOperator(str, val) {
    if (is_1.default.string(str) && comparisonOperators[str]) {
        let op = comparisonOperators[str];
        if (typeof val === 'number' && isNaN(val) && op !== 'EQUAL') {
            throw new Error('Invalid query. You can only perform equals comparisons on NaN.');
        }
        if (val === null && op !== 'EQUAL') {
            throw new Error('Invalid query. You can only perform equals comparisons on Null.');
        }
        return true;
    }
    throw new Error('Operator must be one of "<", "<=", "==", ">", or ">=".');
}
exports.validateComparisonOperator = validateComparisonOperator;
/*!
 * Validates that 'value' is a DocumentReference.
 *
 * @param {*} value The argument to validate.
 * @returns 'true' is value is an instance of DocumentReference.
 */
function validateDocumentReference(value) {
    if (is_1.default.instanceof(value, DocumentReference)) {
        return true;
    }
    throw validate_1.customObjectError(value);
}
exports.validateDocumentReference = validateDocumentReference;
/**
 * Verifies euqality for an array of objects using the `isEqual` interface.
 *
 * @private
 * @param {Array.<Object>} left Array of objects supporting `isEqual`.
 * @param {Array.<Object>} right Array of objects supporting `isEqual`.
 * @return {boolean} True if arrays are equal.
 */
function isArrayEqual(left, right) {
    if (left.length !== right.length) {
        return false;
    }
    for (let i = 0; i < left.length; ++i) {
        if (!left[i].isEqual(right[i])) {
            return false;
        }
    }
    return true;
}
