/*!
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/// <reference types="node" />
import { EventEmitter } from 'events';
import * as r from 'request';
import { ApiError, BodyResponseCallback, DecorateRequestOptions } from './util';
export declare type RequestResponse = [Metadata, r.Response];
export interface ServiceObjectParent {
    Promise?: PromiseConstructor;
    requestStream(reqOpts: DecorateRequestOptions): r.Request;
    request(reqOpts: DecorateRequestOptions, callback: BodyResponseCallback): void;
}
export interface Interceptor {
    request(opts: r.Options): DecorateRequestOptions;
}
export declare type GetMetadataOptions = object;
export declare type Metadata = any;
export declare type MetadataResponse = [Metadata, r.Response];
export declare type MetadataCallback = (err: Error | null, metadata?: Metadata, apiResponse?: r.Response) => void;
export declare type ExistsOptions = object;
export interface ExistsCallback {
    (err: Error | null, exists?: boolean): void;
}
export interface ServiceObjectConfig {
    /**
     * The base URL to make API requests to.
     */
    baseUrl?: string;
    /**
     * The method which creates this object.
     */
    createMethod?: Function;
    /**
     * The identifier of the object. For example, the name of a Storage bucket or
     * Pub/Sub topic.
     */
    id?: string;
    /**
     * A map of each method name that should be inherited.
     */
    methods?: Methods;
    /**
     * The parent service instance. For example, an instance of Storage if the
     * object is Bucket.
     */
    parent: ServiceObjectParent;
}
export interface Methods {
    [methodName: string]: {
        reqOpts?: r.CoreOptions;
    } | boolean;
}
export interface InstanceResponseCallback<T> {
    (err: ApiError | null, instance?: T | null, apiResponse?: r.Response): void;
}
export declare type CreateOptions = {};
export declare type CreateResponse<T> = any[];
export interface CreateCallback<T> {
    (err: ApiError | null, instance?: T | null, ...args: any[]): void;
}
export declare type DeleteOptions = object;
export interface DeleteCallback {
    (err: Error | null, apiResponse?: r.Response): void;
}
export interface GetConfig {
    /**
     * Create the object if it doesn't already exist.
     */
    autoCreate?: boolean;
}
declare type GetOrCreateOptions = GetConfig & CreateOptions;
export declare type GetResponse<T> = [T, r.Response];
export interface ResponseCallback {
    (err?: Error | null, apiResponse?: r.Response): void;
}
export declare type SetMetadataResponse = [Metadata];
export declare type SetMetadataOptions = object;
/**
 * ServiceObject is a base class, meant to be inherited from by a "service
 * object," like a BigQuery dataset or Storage bucket.
 *
 * Most of the time, these objects share common functionality; they can be
 * created or deleted, and you can get or set their metadata.
 *
 * By inheriting from this class, a service object will be extended with these
 * shared behaviors. Note that any method can be overridden when the service
 * object requires specific behavior.
 */
declare class ServiceObject<T = any> extends EventEmitter {
    metadata: Metadata;
    baseUrl?: string;
    parent: ServiceObjectParent;
    id?: string;
    private createMethod?;
    protected methods: Methods;
    protected interceptors: Interceptor[];
    Promise?: PromiseConstructor;
    constructor(config: ServiceObjectConfig);
    /**
     * Create the object.
     *
     * @param {object=} options - Configuration object.
     * @param {function} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {object} callback.instance - The instance.
     * @param {object} callback.apiResponse - The full API response.
     */
    create(options?: CreateOptions): Promise<CreateResponse<T>>;
    create(options: CreateOptions, callback: CreateCallback<T>): void;
    create(callback: CreateCallback<T>): void;
    /**
     * Delete the object.
     *
     * @param {function=} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {object} callback.apiResponse - The full API response.
     */
    delete(options?: DeleteOptions): Promise<[r.Response]>;
    delete(options: DeleteOptions, callback: DeleteCallback): void;
    delete(callback: DeleteCallback): void;
    /**
     * Check if the object exists.
     *
     * @param {function} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {boolean} callback.exists - Whether the object exists or not.
     */
    exists(options?: ExistsOptions): Promise<[boolean]>;
    exists(options: ExistsOptions, callback: ExistsCallback): void;
    exists(callback: ExistsCallback): void;
    /**
     * Get the object if it exists. Optionally have the object created if an
     * options object is provided with `autoCreate: true`.
     *
     * @param {object=} options - The configuration object that will be used to
     *     create the object if necessary.
     * @param {boolean} options.autoCreate - Create the object if it doesn't already exist.
     * @param {function} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {object} callback.instance - The instance.
     * @param {object} callback.apiResponse - The full API response.
     */
    get(options?: GetOrCreateOptions): Promise<GetResponse<T>>;
    get(callback: InstanceResponseCallback<T>): void;
    get(options: GetOrCreateOptions, callback: InstanceResponseCallback<T>): void;
    /**
     * Get the metadata of this object.
     *
     * @param {function} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {object} callback.metadata - The metadata for this object.
     * @param {object} callback.apiResponse - The full API response.
     */
    getMetadata(options?: GetMetadataOptions): Promise<MetadataResponse>;
    getMetadata(options: GetMetadataOptions, callback: MetadataCallback): void;
    getMetadata(callback: MetadataCallback): void;
    /**
     * Set the metadata for this object.
     *
     * @param {object} metadata - The metadata to set on this object.
     * @param {object=} options - Configuration options.
     * @param {function=} callback - The callback function.
     * @param {?error} callback.err - An error returned while making this request.
     * @param {object} callback.apiResponse - The full API response.
     */
    setMetadata(metadata: Metadata, options?: SetMetadataOptions): Promise<SetMetadataResponse>;
    setMetadata(metadata: Metadata, callback: MetadataCallback): void;
    setMetadata(metadata: Metadata, options: SetMetadataOptions, callback: MetadataCallback): void;
    /**
     * Make an authenticated API request.
     *
     * @private
     *
     * @param {object} reqOpts - Request options that are passed to `request`.
     * @param {string} reqOpts.uri - A URI relative to the baseUrl.
     * @param {function} callback - The callback function passed to `request`.
     */
    private request_;
    /**
     * Make an authenticated API request.
     *
     * @private
     *
     * @param {object} reqOpts - Request options that are passed to `request`.
     * @param {string} reqOpts.uri - A URI relative to the baseUrl.
     * @param {function} callback - The callback function passed to `request`.
     */
    request(reqOpts: DecorateRequestOptions): Promise<RequestResponse>;
    request(reqOpts: DecorateRequestOptions, callback: BodyResponseCallback): void;
    /**
     * Make an authenticated API request.
     *
     * @private
     *
     * @param {object} reqOpts - Request options that are passed to `request`.
     * @param {string} reqOpts.uri - A URI relative to the baseUrl.
     */
    requestStream(reqOpts: DecorateRequestOptions): r.Request;
}
export { ServiceObject };
