/*! firebase-admin v6.0.0 */
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var error_1 = require("../utils/error");
var validator = require("../utils/validator");
/**
 * Internals of a Database instance.
 */
var DatabaseInternals = /** @class */ (function () {
    function DatabaseInternals() {
        this.databases = {};
    }
    /**
     * Deletes the service and its associated resources.
     *
     * @return {Promise<()>} An empty Promise that will be fulfilled when the service is deleted.
     */
    DatabaseInternals.prototype.delete = function () {
        for (var _i = 0, _a = Object.keys(this.databases); _i < _a.length; _i++) {
            var dbUrl = _a[_i];
            var db = this.databases[dbUrl];
            db.INTERNAL.delete();
        }
        return Promise.resolve(undefined);
    };
    return DatabaseInternals;
}());
var DatabaseService = /** @class */ (function () {
    function DatabaseService(app) {
        this.INTERNAL = new DatabaseInternals();
        if (!validator.isNonNullObject(app) || !('options' in app)) {
            throw new error_1.FirebaseDatabaseError({
                code: 'invalid-argument',
                message: 'First argument passed to admin.database() must be a valid Firebase app instance.',
            });
        }
        this.appInternal = app;
    }
    Object.defineProperty(DatabaseService.prototype, "app", {
        /**
         * Returns the app associated with this DatabaseService instance.
         *
         * @return {FirebaseApp} The app associated with this DatabaseService instance.
         */
        get: function () {
            return this.appInternal;
        },
        enumerable: true,
        configurable: true
    });
    DatabaseService.prototype.getDatabase = function (url) {
        var dbUrl = this.ensureUrl(url);
        if (!validator.isNonEmptyString(dbUrl)) {
            throw new error_1.FirebaseDatabaseError({
                code: 'invalid-argument',
                message: 'Database URL must be a valid, non-empty URL string.',
            });
        }
        var db = this.INTERNAL.databases[dbUrl];
        if (typeof db === 'undefined') {
            var rtdb = require('@firebase/database');
            var version = require('../../package.json').version;
            db = rtdb.initStandalone(this.appInternal, dbUrl, version).instance;
            this.INTERNAL.databases[dbUrl] = db;
        }
        return db;
    };
    DatabaseService.prototype.ensureUrl = function (url) {
        if (typeof url !== 'undefined') {
            return url;
        }
        else if (typeof this.appInternal.options.databaseURL !== 'undefined') {
            return this.appInternal.options.databaseURL;
        }
        throw new error_1.FirebaseDatabaseError({
            code: 'invalid-argument',
            message: 'Can\'t determine Firebase Database URL.',
        });
    };
    return DatabaseService;
}());
exports.DatabaseService = DatabaseService;
